#!/bin/env python3
import hashlib
import flask
import bdd
import  server_common as common
import server_api

# Code écrit par Laurent Signac / Jeanvier 2017

TPL = flask.render_template
SITEKEY = "isnsite2016"


app = flask.Flask(__name__)

app.register_blueprint(server_api.restapi)
# ===========================================================================
# Site  Web
# ============================================================================

# Pb CORS constaté avec client Javascript  : https://en.wikipedia.org/wiki/Cross-origin_resource_sharing
@app.errorhandler(404)
def error_404(_):
    stri = "Page not found"
    rep = flask.make_response(stri)
    rep.headers['Access-Control-Allow-Origin'] = '*'
    # print(rep)
    # print(dir(rep))
    return rep, 404

@app.route('/info')
@app.route('/')
def info():
    data = """\
    <p>Bonjour visiteur,
    </p>
    <p>
    Pour utiliser le service, vous devez vous connecter
    </p>
    """
    return TPL("default.html", title='Home (non connecté)', data=data)

@app.route('/home')
def home():
    login = common.check_login()
    data = """\
    <p>Bonjour visiteur {},
    </p>
    <p>
    Pour utiliser le service, obtenez une clé (menu Nouvelle clé)
    que vous utiliserez dans le client.
    </p>
    <p> 
    Vous pouvez aussi visualiser le tableau des scores, et la liste 
    des graphes proposés.
    </p>
    """.format(login)
    return TPL("default.html", title='Home', data=data)

@app.route('/error_new_account')
def error_new_account():
    return TPL("default.html", title='Erreur', data='Problème à la création du compte')

@app.route('/new_account', methods=['POST', 'GET'])
def new_account():
    if flask.request.method == 'POST':
        login = flask.request.form['login']
        password = flask.request.form['password']
        sitekey = flask.request.form['sitekey']
        if sitekey != SITEKEY:
            return flask.redirect(flask.url_for('error_new_account'))
        hashpass = hashlib.sha256("G2016-{}{}".format(login, password).encode('utf8')).hexdigest()
        if bdd.new_account(login, hashpass):
            return flask.redirect(flask.url_for('login_cb'))
        else:
            return flask.redirect(flask.url_for('error_new_account'))

    data = '''\
        <form action="" class="form" method="post">
        <div class='form-group'>
            <label for='login'>Login</label>
            <input type=text class='form-control' name='login'>
        </div>
        <div class='form-group'>
            <label for='password'>Password</label>
            <input type=password class='form-control' name='password'>
        </div>
         <div class='form-group'>
             <label for='sitekey'>Clé du site</label>
             <input type=password class='form-control' name='sitekey'>
         </div>
        <button type="submit" class="btn btn-default">Créer un compte</button>
        </form>
    '''
    return TPL("default.html", title='Créer un compte', data=data)



@app.route('/get_newkey', methods=['GET'])
def get_newkey():
    login = common.check_login()
    key = common.store_new_key(login)
    return TPL("default.html", title="Clé utilisateur {}".format(login), data="Clé : {}".format(key))


@app.route('/login', methods=['GET', 'POST'])
def login_cb():
    if flask.request.method == 'POST':
        login_ = flask.request.form['username']
        password = flask.request.form['password']
        hashpass = hashlib.sha256("G2016-{}{}".format(login_, password).encode('utf8')).hexdigest()
        if not bdd.check_account(login_, hashpass):
            return flask.redirect(flask.url_for('login_cb'))
        else:
            flask.session['username'] = login_
            return flask.redirect(flask.url_for('home'))
    data = '''\
        <form action="" class="form" method="post">
        <div class='form-group'>
            <label for="username">Login : </label>
            <input class="form-control" type=text name=username>
        </div><div class='form-group'>
            <label for="password">MdPasse : </label>
            <input class="form-control" type=password name=password>
        </div><div class='form-group'>
            <button type="submit" class="btn btn-default">Se connecter</button>
        </div>
        </form>
    '''
    return TPL("default.html", title='Login page', data=data)

@app.route('/logout')
def logout():
    # remove the username from the session if it's there
    flask.session.pop('username', None)
    return flask.redirect(flask.url_for('info'))

@app.route('/graph/<name>')
def graph(name):
    common.check_login()
    if name in common.GAMES:
        return TPL("default.html", title=name, data='<b>{}</b>\n<pre>{}</pre>'.format(name, common.GAMES[name]['graph']))
    else:
        return TPL("default.html", title='Graphe inconnu')

@app.route('/list_games')
def list_games():
    common.check_login()
    lst = ['<ul>']
    for name, g in sorted(common.GAMES.items()):
        lst.append("<li><a href={}>{}</a> : Graphe à {} noeuds et {} aretes</li>".format(\
                  flask.url_for('graph', name=name), name, g['graph'].nb_nodes(), g['graph'].nb_edges()))
    lst.append('</ul>')
    return TPL("default.html", data="\n".join(lst), title="Liste des jeux")

@app.route('/scores')
def scores():
    common.check_login()
    lst = ['<ul>']
    for name in sorted(common.GAMES):
        res = bdd.get_scores_graph(name)
        res2 = ["<li>{} : {}</li>".format(login, score) for login, score in res]
        if res2:
            lst.append("<li>{}<ul>".format(name))
            lst.extend(res2)
            lst.append("</ul/</li>")

    lst.append('</ul>')
    return TPL("scores.html", scores="\n".join(lst))


# ==========================================================================================
# MAIN
# ==========================================================================================
common.create_games()
app.secret_key = 'Abdsqk23*lezneri123445'
print("PATH =====>", app.instance_path)
bdd.create_db(app.instance_path+"/graphe_game.db")
if __name__ == '__main__':
    # app.config['DEBUG'] = True
    # set the secret key.  keep this really secret:
    app.run(host='0.0.0.0')
