import math
import random
import itertools

# Code écrit par Laurent Signac, Janvier 2017
# Graphe
# =================================================================
class Graphe:
    """ Classe permettant de représenter des graphes non orientés
        chaque noeud peut etre nommé et avoir une valeur
        En revanche, les coordonnées des noeuds ne sont pas stockées
        dans le graphe (choix de design).
    """

    def __init__(self, i_edges, i_values, default_value=0):
        """ Initialisation d'un graphe :
        g = Graphe(["AB", "AC", ("A", "TITI")], {'A': 3, 'TITI': 5})
        Les étiquettes des noeuds ne doivent pas contenir -
        """

        # Ensemble des noeuds mentionnés dans i_edges, i_values
        nodes = set()
        for n in i_values:
            nodes.add(n)
        for edge in i_edges:
            for n in edge:
                nodes.add(n)
        for n in nodes:
            if "-" in n:
                raise ValueError("Nodes names must not contain -")
        # Structure de données pour un graphe
        graphe = dict()
        values = dict()
        for n in nodes:
            graphe[n] = set()
            values[n] = default_value
        # Remplissage de la structure
        for n1, n2 in i_edges:
            graphe[n1].add(n2)
            graphe[n2].add(n1)

        for k, v in i_values.items():
            values[k] = v
        self._graphe = graphe
        self._values = values

    def __len__(self):
        return len(self._graphe)

    def __contains__(self, x):
        return x in self._graphe

    def __str__(self):
        l = []
        for node, edges in sorted(self._graphe.items()):
            l.append("{} : {} -> {}".format(node, self._values[node], sorted(edges)))
        return "\n".join(l)

    def __repr__(self):
        tuples = []
        for arete in self.iter_edges():
            tuples.append(arete)
        return "Graphe({}, {})".format(tuples, self._values)

    def __iter__(self):
        return self.iter_nodes()

    def __copy__(self):
        return Graphe(list(self.iter_edges()), self._values)

    def __getitem__(self, nodename):
        """ Permet la notation graphe["A"] pour avoir
            la valeur du noeud A
        """
        return self._values[nodename]

    def __bool__(self):
        """ Permet d'écrire : if graphe: ...
            un graphe est Vrai s'il contient au moins un noeud
        """
        return bool(self._values)

    # Plongements
    # ==================================================================

    def embed(self):
        """ Plongement dans le plan : les noeuds sont placés en cercle
        """

        n = self.nb_nodes()
        plong = dict()
        for i, node in enumerate(self._graphe):
            x = math.cos(i*2*math.pi / n)
            y = math.sin(i*2*math.pi / n)
            plong[node] = (x, y)
        return plong

    def copy(self):
        """ Renvoie une copie du graphe """
        return self.__copy__()

    def iter_nodes(self):
        """ Itérateur sur les noeuds
        for n in graphe.iter_nodes():
            print(n)
        """
        for node in sorted(self._graphe):
            yield node

    def iter_values(self):
        """ Itérateur sur les valeurs des noeuds
        for name, value in graphe.iter_values():
            print(name, value)
        """
        for node in sorted(self._graphe):
            yield node, self[node]

    def iter_edges(self, node_=None):
        """ Itérateur sur les aretes :
        for a in graphe.iter_edges():
            print(a)
        """
        if node_ is None:
            for node, edges in sorted(self._graphe.items()):
                for edge in sorted(edges):
                    if node < edge:
                        yield node, edge
        else:
            for edge in self._graphe[node_]:
                yield edge


    def neighbours(self, node):
        """ Renvoie la liste des voisins d'un noeud """
        return self._graphe[node].copy()

    def value(self, node):
        """  Propriété : valeur d'un noeud """
        return self._values[node]

    def remove(self, node):
        """ Enlève un noeud et toutes les aretes
        qui l'ont pour extrémité
        """
        l_edg = []
        for v in self.neighbours(node):
            if v != node:
                self._graphe[v].remove(node)
                l_edg.append((node, v))
        del self._graphe[node]
        del self._values[node]
        return [node], l_edg

    def deep_remove(self, node):
        """ Enleve un noeud et tous ses voisins
        """
        vois = self.neighbours(node)
        l_nod, l_edg = self.remove(node)
        for v in vois:
            n, e = self.remove(v)
            l_nod.extend(n)
            l_edg.extend(e)
        return l_nod, l_edg

    def nb_nodes(self):
        """ Renvoie le nombre de noeuds"""
        return len(self._graphe)

    def nb_edges(self, node=None):
        """ Renvoie le nombre d'aretes """
        if node is None:
            s = 0
            for _nodes in self._graphe.values():
                s += len(_nodes)
            return s // 2
        else:
            if node not in self._graphe:
                return 0
            return len(self._graphe[node])


############ FIN DE LA CLASSE Graphe

def random_graph(n, a):
    """ Renvoie un graphe aléatoire
    n : nombre de noeuds
    a : nombre max d'aretes
    """
    node_names = list("ABCDEFGHIJKLMNOPQRSTUVWXYZ")
    l_nodes = node_names.copy()
    for n1, n2 in itertools.product(node_names, repeat=2):
        l_nodes.append(n1+n2)
        if len(l_nodes) > n:
            break

    noeuds = sorted(set(l_nodes[:n]))
    values = {n: random.randint(1, 20) for n in noeuds}
    lst = []
    for _ in range(a):
        arete = random.sample(noeuds, 2)
        lst.append(arete)

    return Graphe(lst, values)

if __name__ == '__main__':
    graphe = Graphe(("AB", "AC", "BC", "AD", "DC", "ED", "EB", "BE"), {"A": 7, "B": 10, "Z": 4})
