#!/usr/bin/env python3
import tkinter as tk
from tkinter import messagebox
import urllib.request
import urllib.parse
import json
import hashlib
import graphe
# import random
# import traceback
# import sys

# Code écrit par Laurent Signac / Jnvier 2017

# Identifiants de connexion et serveur.
# Ces valeurs sont mises à jour si on active le bouton Connexion de l'interface
CREDENTIALS = {"key": "06064a1f-e01e-4e0b-b0c0-e5541b74f45e",
               "login": "invite",
               "url": "http://localhost:5000"}

# CREDENTIALS["url"] = "http://192.168.80.13:5000"
# CREDENTIALS["url"] = "http://lsignac.pythonanywhere.com/" # L'appli en ligne sera supprimée sous peu


# Fonction utilitaires qui font le lien entre les tags de tkinter et
# les noms des noeuds ou aretes

def get_node_with_tags(list_of_tags):
    for tag in list_of_tags:
        if tag.startswith("node-"):
            return tag[5:]
    return ""


def get_edge_with_tags(list_of_tags):
    for tag in list_of_tags:
        if tag.startswith("edge-"):
            nodes = tag.split("-")
            return nodes[1], nodes[2]
    return tuple()


def get_tag_with_node(node):
    return "node-{}".format(node)


def get_tag_with_edge(node1, node2):
    return "edge-{}-{}".format(*sorted((node1, node2)))


# =====================================================================================
# Fenetre de login avec LA CLÉ du compte. Met à jour la structure globale CREDENTIALS
# =====================================================================================
class LoginDialogKey(tk.Toplevel):
    """ Fenetre modale de connexion.
        Si on se connecte avec les bons identifiants,
        la fenetre se referme et les donnés de connexion sont
        enregistrées dans la structure CREDENTIALS.
        Sinon, la fenetre reste affichée.
    """
    def __init__(self, parent):
        tk.Toplevel.__init__(self)
        self.title("Connexion à un serveur de jeu")
        # Pour bien faire, il faudrait aussi ajouter
        # la possibilité de choisir l'URL du serveur
        tk.Label(self, text="Serveur : ").pack()
        self.entry_url_server = tk.Entry(self)
        self.entry_url_server.pack()
        self.entry_url_server.insert(0, CREDENTIALS["url"])
        tk.Label(self, text="Login : ").pack()
        self.entry_login = tk.Entry(self)
        self.entry_login.insert(0, CREDENTIALS["login"])
        self.entry_login.pack()
        tk.Label(self, text="Clé (cf site) : ").pack()
        self.entry_key = tk.Entry(self)   # , show='*')
        self.entry_key.insert(0, CREDENTIALS["key"])
        self.entry_key.pack()
        self.txt = tk.Label(self, text="")
        self.txt.pack()
        tk.Button(self, text="Connexion", command=self.connect).pack()

    def connect(self):
        login = self.entry_login.get().strip()

        key = self.entry_key.get().strip()
        urlserver = self.entry_url_server.get()

        CREDENTIALS["url"] = urlserver
        CREDENTIALS["key"] = key
        CREDENTIALS["login"] = login
        print("NEW CREDENTIALS : ", CREDENTIALS)
        self.destroy()

# ===============================================================
# Fenetre de jeu principale
# ===============================================================
class Fenetre(tk.Frame):
    def __init__(self, parent):
        # Initialiseur de la classe parente :
        super().__init__(parent)

        parent.title("Décimation de graphe")
        self.pack(fill='both', expand=1)

        barre = tk.Frame(self)
        barre.pack(fill='both', expand=1)

        bouton_new = tk.Button(barre, width=10, height=1, text="Nouvelle partie", command=self.new_game)
        bouton_new.pack(side="right")
        bouton_new = tk.Button(barre, width=10, height=1, text="Connexion", command=self.connect)
        bouton_new.pack(side="left")
        self.canvas = tk.Canvas(self, width=500, height=500, background="grey")
        self.canvas.pack(padx=8, pady=8)  # Ajouter  fill='both', expand=1 pour voir...

        self.score = 0
        self.graph = None
        #self.values_n = {}
        self.graphname = ""
        self.positions = None
        self.game = []

    # Callbacks associées aux boutons
    # =====================================================
    @staticmethod
    def connect():
        """ Callback de connexion. Ouvre une fenetre modale """
        result = LoginDialogKey(root)  # Remplacer par LoginDialog pour changer de mode de connexion
        result.transient(root)
        result.grab_set()
        root.wait_window(result)

    def new_game(self):
        """ Obtention d'une nouvelle partie. Utilise la structure
            CREDENTIALS pour se connecter
        """
        login = CREDENTIALS["login"]
        key = CREDENTIALS["key"]
        url = CREDENTIALS["url"]
        print("URL : ", url + "/get_game/{}/{}".format(login, key))
        data_json = urllib.request.urlopen(url + "/get_game/{}/{}".format(login, key)).read().decode('utf8')
        data = json.loads(data_json)
        print("RETOUR : ", data)
        if data["status"] == 0:
            messagebox.showwarning("Erreur", data["message"])
        else:
            self.new_graph(data['name'], graphe.Graphe(data["graph_edges"], data['values']), data.get('positions', None))

    def post_game(self):
        """ Envoi des résultats de la partie au serveur (se fait automatiquement
            lorsqu'on termine une partie).
            Utilise CREDENTIALS pour se connecter
        """
        login = CREDENTIALS["login"]
        key = CREDENTIALS["key"]
        url = CREDENTIALS["url"]
        data = urllib.parse.urlencode({'name': self.graphname, 'game': json.dumps(self.game)}).encode('utf8')
        print({'name': self.graphname, 'game': json.dumps(self.game)})
        res = urllib.request.urlopen(url + "/post_game/{}/{}".format(login, key), data=data).read().decode('utf8')
        return json.loads(res)

    # Callbacks associées à la souris : clic droit sur un noeud (delete)
    # et drag sur un noeud ou texte (move)
    # ======================================================================
    def delete_node(self, event):
        """ Efface le noeud (selon les règles) actuellement dans tk.CURRENT
            (de la structure de graphe et de l'écran)
        """
        # num = tk.CURRENT
        node = get_node_with_tags(self.canvas.gettags(tk.CURRENT))
        self.score = self.score + self.graph[node]
        l_nodes, l_edges = self.graph.deep_remove(node)
        self.game.append(node)

        for n in l_nodes:
            self.canvas.delete(get_tag_with_node(n))

        for e in l_edges:
            self.canvas.delete(get_tag_with_edge(*e))

        # print(self.graph)
        print("Score : ", self.score)
        # Si on a gagné (si le graphe est vide)
        if not self.graph:
            print("Score FINAL =======>  ", self.score)
            print("PARTIE         ====>", self.game)
            res = self.post_game()
            if res['status'] == 0:
                titre = "Erreur"
            else:
                titre = "Bravo!"
            messagebox.showinfo(titre, res['message'], parent=gui)

    def move(self, event):
        def node_center(node_):
            """ Renvoie le centre du noeud étant donné
                son rectangle englobant
            """
            tag = get_tag_with_node(node_)
            x1_, y1_, x2_, y2_ = self.canvas.coords(tag)
            return (x1_ + x2_) // 2, (y1_ + y2_) // 2

        x, y = event.x, event.y
        # num = tk.CURRENT
        node = get_node_with_tags(self.canvas.gettags(tk.CURRENT))
        x1, y1 = node_center(node)
        for v in self.graph.neighbours(node):
            edge = get_tag_with_edge(node, v)
            x2, y2 = node_center(v)
            self.canvas.coords(edge, x1, y1, x2, y2)
        self.canvas.move(get_tag_with_node(node), x - x1, y - y1)

    # Utilitaires
    # ================================================================
    def new_graph(self, name, graphe_, positions):
        """
        Entrée : - nom du graphe
                 - un graphe
                 - un dictionnaire avec les positions des noeuds (ou None si on n'en a pas)

        Affiche le nouveau graphe dans la fenetre pour qu'on puisse
        jouer la partie
        """
        # S'il y a dejà des objets graphiques, il faut les supprimer
        for idobj in self.canvas.find_all():
            self.canvas.delete(idobj)

        self.graph = graphe_
        self.positions = positions
        self.draw_graph()

        self.score = 0
        self.graphname = name
        self.game = []  # Liste des coups joués
        self.master.title(name)

    # def clickobj(self, event):
    #    print(event.widget.find_closest(event.x, event.y))

    def draw_graph(self):
        """ Affiche le graphe courant
            en utilisant le plongement fourni s'il y en a un
            et un plongement standard sinon
        """

        def convert_coord(x_, y_):
            """ Convertit des coordonnées graphe (-1 à 1)
                en coordonnées canvas (selon la taille du canvas)
            """
            multx = int(self.canvas.winfo_width() * 2 / 5)
            multy = int(self.canvas.winfo_height() * 2 / 5)
            offx = self.canvas.winfo_width() // 2
            offy = self.canvas.winfo_height() // 2
            return x_ * multx + offx, y_ * multy + offy

        graph = self.graph
        if self.positions is not None:
            plong = self.positions
            print("PLONGEMENT FOURNI")
        else:
            plong = graph.embed()

        # Affichage des aretes
        for n1, n2 in graph.iter_edges():
            x1, y1 = plong[n1]
            x1, y1 = convert_coord(x1, y1)
            x2, y2 = plong[n2]
            x2, y2 = convert_coord(x2, y2)
            tmp = self.canvas.create_line(x1, y1, x2, y2, fill="#ff0000")
            self.canvas.addtag_withtag(get_tag_with_edge(n1, n2), tmp)

        # Affichage des noeuds et du texte
        for node in graph:
            x, y = plong[node]
            x, y = convert_coord(x, y)
            tmpnode = self.canvas.create_oval(x - 10, y - 10, x + 10, y + 10, fill="#000066", outline="#0000ff")
            tmptxt = self.canvas.create_text(x, y, fill="#ffffff", text=str(graph[node]))
            for tmpid in tmpnode, tmptxt:
                self.canvas.addtag_withtag(get_tag_with_node(node), tmpid)
                self.canvas.tag_bind(tmpid, '<B1-Motion>', self.move)
                self.canvas.tag_bind(tmpid, '<Button-3>', self.delete_node)


root, gui = None, None


def main():
    global root, gui
    # Création de l'application
    root = tk.Tk()
    # Ajout des éléments graphiques
    gui = Fenetre(root)
    # Boucle des événements
    root.mainloop()

if __name__ == "__main__":
    main()
